﻿using gov.va.med.vbecs.Common;
using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

namespace gov.va.med.vbecs.BOL
{
    /// <summary>
    /// Create Exception Reports
    /// </summary>
    public static class ExceptionReportCreator
    {
        /// <summary>
        /// Validate and Create exception reports for pending test acceptance
        /// </summary>
        /// <param name="pendingTestList"></param>
        /// <returns></returns>
        public static IList<ExceptionReport> CreateExceptionReports(IEnumerable<IPendingTestModel> pendingTestList)
        {
            var exceptionList = new List<ExceptionReport>();

            foreach (var pendingTest in pendingTestList)
            {
                if (pendingTest.TestValidation.ExceptionForDisplayList != null &&
                    pendingTest.TestValidation.ExceptionForDisplayList.Any())
                {
                    if (pendingTest.TestValidation.ExceptionForDisplayList.Any(x => string.IsNullOrWhiteSpace(x.OverrideMessage)))
                    {
                        throw new Exception("Exception override comment is required for all exceptions");
                    }

                    foreach (var exception in pendingTest.TestValidation.ExceptionForDisplayList)
                    {
                        var exceptionReport = new ExceptionReport(exception.ExceptionType, exception.OverrideMessage, Common.LogonUser.LogonUserName);

                        // Generate exception base on exception type.
                        if (exception.ExceptionType == ExceptionType.Unit_ER_issued_testing_problem)
                        {
                            exceptionList.AddRange(GenerateExceptionEmergencyIssueUnitTest(pendingTest, exceptionReport));
                        }
                        else if (exception.ExceptionType == ExceptionType.ABORh_confirmation_inconclusive_unit_quarantined ||
                                 exception.ExceptionType == ExceptionType.ABORh_confirmation_does_not_match_login_ABORh)
                        {
                            exceptionList.Add(GenerateExceptionUnitAboRh(pendingTest, exceptionReport));
                        }
                        else if (exception.ExceptionType == ExceptionType.ABORh_discrepancy)
                        {
                            exceptionList.Add(GenerateExceptionPatientAboRhDiscrepancy(pendingTest, exceptionReport));
                        }
                        else if (exception.ExceptionType == ExceptionType.Expired_task_processed)
                        {
                            exceptionList.Add(GenerateExceptionExpiredTaskProcessed(pendingTest, exceptionReport));
                        }
                        else if (exception.ExceptionType == ExceptionType.Discrepant_patient_antigen_typing)
                        {
                            exceptionList.Add(GenerateExceptionDiscrepantPatientAntigenTyping(pendingTest, exceptionReport));
                        }
                        else if (exception.ExceptionType == ExceptionType.Crossmatch_incompatible_Give_only_with_MD_approval)
                        {
                            exceptionList.Add(GenerateExceptionCrossMatchIncompatibleGiveWithMDApproval(pendingTest as PendingSpecimenXMatchModel, exceptionReport));
                        }
                        else
                        {
                            throw new Exception("Exception override report cannot be generated.");
                        }
                    }
                }
            }

            return exceptionList;
        }

        /// <summary>
        /// Generate exception for ExceptionType.Crossmatch_incompatible_Give_only_with_MD_approval
        /// </summary>
        /// <param name="pendingXMTest"></param>
        /// <param name="exceptionReport"></param>
        /// <returns></returns>
        private static ExceptionReport GenerateExceptionCrossMatchIncompatibleGiveWithMDApproval(PendingSpecimenXMatchModel pendingXMTest, ExceptionReport exceptionReport)
        {
            if (pendingXMTest == null)
            {
                throw new Exception("Pending test cannot be convered to PendingSpecimenXMatchModel to GenerateExceptionUnitSelection.");
            }
            else if (pendingXMTest.OrderedUnitModel == null)
            {
                throw new Exception("Pending XM test cannot have null OrderedUnitModel (cannot generate GenerateExceptionUnitSelection).");
            }
            else if (pendingXMTest.OrderedUnitModel.SelectedDate == null)
            {
                throw new Exception("Pending XM test cannot have null OrderedUnitModel.SelectedDate (cannot generate GenerateExceptionUnitSelection).");
            }
            else
            {
                if (pendingXMTest.BloodUnitModel == null)
                {
                    throw new Exception("Pending blood unit does not have a blood unit specified (Cannot GenerateExceptionUnitSelection).");
                }

                Patient patient = new BOL.Patient(pendingXMTest.PatientGuid);

                exceptionReport.GenerateExceptionUnitSelection(pendingXMTest.BloodUnitModel.BloodUnitGuid,
                                                               patient,
                                                               pendingXMTest.OrderedUnitModel.SelectedDate.Value,
                                                               pendingXMTest.InstrumentUserId);
            }

            return exceptionReport;
        }

        /// <summary>
        /// GenerateExceptionDiscrepantPatientAntigenTyping
        /// </summary>
        /// <param name="pendingTest"></param>
        /// <param name="exceptionReport"></param>
        /// <returns></returns>
        private static ExceptionReport GenerateExceptionDiscrepantPatientAntigenTyping(IPendingTestModel pendingTest, ExceptionReport exceptionReport)
        {
            var pendingSpecimen = pendingTest as IPendingSpecimenTestModel;

            if (pendingSpecimen == null)
            {
                throw new Exception("Pending test cannot be convered to IPendingSpecimenTestModel to GenerateExceptionDiscrepantPatientAntigenTyping.");
            }
            //*** Fortify Justified Code ***
            //pendingSpecimen can't be null since it is checked earler (see above). 
            if (pendingSpecimen.OrderedTest == null)
            {
                throw new Exception("Pending specimen does not have a matching OrderedTest (Cannot GenerateExceptionDiscrepantPatientAntigenTyping).");
            }
            //*** Fortify Justified Code ***

            DataTable dtHistoricalResults = pendingSpecimen.OrderedTest.GetHistoricalTestResultsForOrderedTest();

            if (dtHistoricalResults.Rows.Count > 0)
            {
                DataRow[] dtHistoricalInterps = dtHistoricalResults.Select("BloodTestName LIKE '*Interp*'");
                if (dtHistoricalInterps.Length > 0)
                {
                    foreach (DataRow drHistoricalInterp in dtHistoricalInterps)
                    {
                        var interp = pendingTest.TestResultList.FirstOrDefault(x => x.IsInterp);

                        if (interp.TestResultId != drHistoricalInterp[TABLE.TestResult.TestResultId].ToString().Trim() &&
                            drHistoricalInterp[TABLE.TestResult.TestResultId].ToString().Trim() != string.Empty)
                        {
                            //Defect 309540
                            string reagentTypeName = pendingSpecimen.OrderedTest.OrderableTest.Replace("Repeat", "");
                            reagentTypeName = reagentTypeName.Contains("AGwD")
                                ? "Anti-D"
                                : reagentTypeName.Replace("AG", "Anti-").Trim();
                            exceptionReport.GenerateExceptionPatientAntigenDiscrepancy(pendingSpecimen.OrderedTest,
                                                                             reagentTypeName, 
                                                                             string.Empty,
                                                                             string.Empty,
                                                                             string.Empty,
                                                                             null,
                                                                             drHistoricalInterp[TABLE.TestResult.TestResultId].ToString().Trim(),
                                                                             interp.TestResultId,
                                                                             pendingTest.InstrumentUserId);
                            break;
                        }
                    }
                }
            }

            return exceptionReport;
        }

        /// <summary>
        /// GenerateExceptionExpiredTaskProcessed
        /// </summary>
        /// <param name="pendingTest"></param>
        /// <param name="exceptionReport"></param>
        /// <returns></returns>
        private static ExceptionReport GenerateExceptionExpiredTaskProcessed(IPendingTestModel pendingTest, ExceptionReport exceptionReport)
        {
            var pendingSpecimen = pendingTest as IPendingSpecimenTestModel;
            OrderedTest orderedTest = null;
            OrderedComponent orderedComponent = null;

            if (pendingSpecimen == null)
            {
                throw new Exception("Pending test cannot be convered to IPendingSpecimenTestModel to GenerateExceptionExpiredTaskProcessed.");
            }
            //*** Fortify Justified Code ***
            //pendingSpecimen can't be null since it is checked earler (see above). 
            orderedTest = pendingSpecimen.OrderedTest;
            //*** Fortify Justified Code ***

            // This might be a XM. Check for it's OrderedComponent
            var pendingXM = pendingTest as PendingSpecimenXMatchModel;

            if (pendingXM != null &&
                pendingXM.OrderedComponentGuid.HasValue)
            {
                orderedComponent = new BOL.OrderedComponent(pendingXM.OrderedComponentGuid.Value);
            }

            if (orderedTest == null &&
                orderedComponent == null)
            {
                throw new Exception("Pending test does not have a matching OrderedTest or OrderedComponent (Cannot GenerateExceptionExpiredTaskProcessed).");
            }
            
            exceptionReport.GenerateExceptionExpiredTask(orderedTest, orderedComponent, pendingSpecimen.InstrumentUserId);

            return exceptionReport;
        }

        /// <summary>
        /// BR_92.06
        /// </summary>
        /// <param name="pendingTest"></param>
        /// <param name="exceptionReport"></param>
        /// <returns></returns>
        private static ExceptionReport GenerateExceptionPatientAboRhDiscrepancy(IPendingTestModel pendingTest, ExceptionReport exceptionReport)
        {
            var pendingSpecimen = pendingTest as PendingSpecimenAboRhModel;
            if (pendingSpecimen == null)
            {
                throw new Exception("Pending test cannot be convered to PendingSpecimenAboRhModel to GenerateExceptionPatientAboRhDiscrepancy.");
            }
            //*** Fortify Justified Code ***
            //pendingSpecimen can't be null since it is checked earler (see above). 
            if (pendingSpecimen.OrderedTest == null)
            {
                throw new Exception("Pending specimen does not have a matching OrderedTest (Cannot GenerateExceptionPatientAboRhDiscrepancy).");
            }
            //*** Fortify Justified Code ***

            var testAbo = pendingSpecimen.AboInterp.TestResultId;
            var testRh = pendingSpecimen.RhInterp.TestResultId;
            var previousAboInterp = "";
            var previousRhInterp = "";
            var dtPreviousResults = new DataTable();
            var patient = pendingSpecimen.OrderedTest.Patient;

            if (pendingSpecimen.OrderableTest.IsRepeat &&
                pendingSpecimen.OrderedTest.CheckForRepeatAboRhDiscrepancy(testAbo, testRh, ref previousAboInterp, ref previousRhInterp, ref dtPreviousResults) == true)
            {
                DataTable dtAboRhHistory = pendingSpecimen.OrderedTest.Patient.GetPatientAboRHHistory();

                if (dtPreviousResults.Rows.Count > 0)
                {
                    exceptionReport.GenerateExceptionPatientAboRhDiscrepancy(patient,
                        dtPreviousResults.Rows[0][TABLE.PatientSpecimen.SpecimenUid].ToString(),
                        previousAboInterp,
                        previousRhInterp,
                        dtPreviousResults.Rows[0][TABLE.SpecimenTest.TestTechId].ToString(),
                        dtPreviousResults.Rows[0][TABLE.SpecimenTest.EntryTechId].ToString(),
                        dtPreviousResults.Rows[0][TABLE.SpecimenTest.DivisionCode].ToString(),
                        (DateTime)dtPreviousResults.Rows[0][TABLE.SpecimenTest.TestDate],
                        pendingSpecimen.SpecimenUid, testAbo, testRh, pendingSpecimen.InstrumentUserId, Common.LogonUser.LogonUserName,
                        Common.LogonUser.LogonUserDivisionCode, pendingSpecimen.TestedDateTime);
                }
                else
                {
                    exceptionReport.GenerateExceptionPatientAboRhDiscrepancy(patient,
                        "N/A",
                        previousAboInterp,
                        previousRhInterp,
                        "N/A",
                        "N/A",
                        "N/A",
                        DateTime.MinValue,
                        pendingSpecimen.SpecimenUid, testAbo, testRh, pendingSpecimen.InstrumentUserId, Common.LogonUser.LogonUserName,
                        Common.LogonUser.LogonUserDivisionCode, pendingSpecimen.TestedDateTime);
                }
            }
            else
            {
                DataTable dtAboRhHistory = patient.GetPatientAboRHHistory();

                if (dtAboRhHistory.Rows.Count > 0)
                {
                    exceptionReport.GenerateExceptionPatientAboRhDiscrepancy(patient,
                        dtAboRhHistory.Rows[0][TABLE.PatientSpecimen.SpecimenUid].ToString(),
                        pendingSpecimen.OrderedTest.Patient.AboRh.Abo.ToString(),
                        pendingSpecimen.OrderedTest.Patient.AboRh.RHFactorText,
                        dtAboRhHistory.Rows[0][TABLE.SpecimenTest.TestTechId].ToString(),
                        dtAboRhHistory.Rows[0][TABLE.SpecimenTest.EntryTechId].ToString(),
                        dtAboRhHistory.Rows[0][TABLE.SpecimenTest.DivisionCode].ToString(),
                        (DateTime)dtAboRhHistory.Rows[0][TABLE.SpecimenTest.TestDate],
                        pendingSpecimen.SpecimenUid, testAbo, testRh, pendingSpecimen.InstrumentUserId, Common.LogonUser.LogonUserName,
                        Common.LogonUser.LogonUserDivisionCode, pendingSpecimen.TestedDateTime);
                }
                else
                {
                    exceptionReport.GenerateExceptionPatientAboRhDiscrepancy(patient,
                        "N/A",
                        pendingSpecimen.OrderedTest.Patient.AboRh.Abo.ToString(),
                        pendingSpecimen.OrderedTest.Patient.AboRh.RHFactorText,
                        "N/A",
                        "N/A",
                        "N/A",
                        DateTime.MinValue,
                        pendingSpecimen.SpecimenUid, testAbo, testRh, pendingSpecimen.InstrumentUserId, Common.LogonUser.LogonUserName,
                        Common.LogonUser.LogonUserDivisionCode, pendingSpecimen.TestedDateTime);
                }
            }

            return exceptionReport;
        }

        /// <summary>
        /// BR_56.29
        /// </summary>
        /// <param name="pendingTest"></param>
        /// <param name="exceptionReport"></param>
        /// <returns></returns>
        private static IList<ExceptionReport> GenerateExceptionEmergencyIssueUnitTest(IPendingTestModel pendingTest, ExceptionReport exceptionReport)
        {
            var expReportList = new List<ExceptionReport>();

            if (pendingTest is IPendingBloodUnitTestModel)
            {
                var pendingAntigenTypingModel = pendingTest as IPendingBloodUnitTestModel;

                if (pendingAntigenTypingModel == null)
                {
                    throw new Exception("Pending test cannot be convered to PendingBloodUnitAntigenTypingModel to GenerateExceptionEmergencyIssueUnitTest.");
                }
                //*** Fortify Justified Code ***
                //pendingAntigenTypingModel can't be null since it is checked earler (see above). 
                if (pendingAntigenTypingModel.BloodUnitModel == null)
                {
                    throw new Exception("Pending blood unit does not have a blood unit specified (Cannot GenerateExceptionEmergencyIssueUnitTest).");
                }
                //*** Fortify Justified Code ***

                // Get the issued Unit so we can get the PatientGuid
                DataTable issuedUnit = DAL.IssuedUnit.GetIssuedUnitByBloodUnitGuid(pendingAntigenTypingModel.BloodUnitModel.BloodUnitGuid);

                exceptionReport.GenerateExceptionEmergencyIssueUnitTest(pendingAntigenTypingModel.BloodUnitModel.BloodUnitGuid,
                                                                        (Guid)issuedUnit.Rows[0][TABLE.Patient.PatientGuid],
                                                                        pendingAntigenTypingModel.OrderableTest.Name,
                                                                        pendingTest.InterpText,
                                                                        pendingAntigenTypingModel.InstrumentUserId);
                expReportList.Add(exceptionReport);
            }
            else if (pendingTest is IPendingSpecimenTestModel)
            {
                var pendingSpecimenAboRhModel = pendingTest as IPendingSpecimenTestModel;

                if (pendingSpecimenAboRhModel == null)
                {
                    throw new Exception("Pending test cannot be convered to PendingSpecimenAboRhModel to GenerateExceptionEmergencyIssueUnitTest.");
                }
                //*** Fortify Justified Code ***
                //pendingSpecimenAboRhModel can't be null since it is checked earler (see above). 
                if (pendingSpecimenAboRhModel.OrderedTest == null)
                {
                    throw new Exception("Pending specimen does not match an OrderedTest (Cannot GenerateExceptionEmergencyIssueUnitTest).");
                }
                //*** Fortify Justified Code ***

                // Get the issued Unit so we can get the PatientGuid
                DataTable dtIssuedUnits = BOL.OrderedTest.GetEmergencyIssuedUnitsForOrderedTest(pendingSpecimenAboRhModel.OrderedTest.OrderedTestGuid);

                foreach (DataRow drIssuedUnit in dtIssuedUnits.Rows)
                {
                    BOL.ExceptionReport exRep = new BOL.ExceptionReport(exceptionReport.ExceptionType,
                                                                        exceptionReport.ExceptionComment,
                                                                        Common.LogonUser.LogonUserName);

                    exRep.GenerateExceptionEmergencyIssueUnitTest((Guid)drIssuedUnit[TABLE.OrderedUnit.BloodUnitGuid],
                                                                  pendingSpecimenAboRhModel.OrderedTest.PatientGuid,
                                                                  pendingSpecimenAboRhModel.OrderableTest.Name,
                                                                  pendingSpecimenAboRhModel.InterpText,
                                                                  pendingSpecimenAboRhModel.InstrumentUserId);
                    expReportList.Add(exRep);
                }
            }

            return expReportList;
        }

        /// <summary>
        /// BR_2.21, BR_2.17
        /// </summary>
        /// <param name="pendingTest"></param>
        /// <param name="exceptionReport"></param>
        /// <returns></returns>
        private static ExceptionReport GenerateExceptionUnitAboRh(IPendingTestModel pendingTest, ExceptionReport exceptionReport)
        {
            var pendingAboRhBloodUnitModel = pendingTest as PendingBloodUnitAboRhModel;
            if (pendingAboRhBloodUnitModel == null)
            {
                throw new Exception("Pending test cannot be convered to PendingBloodUnitAboRhModel to GenerateExceptionUnitAboRh.");
            }
            //*** Fortify Justified Code ***
            //pendingAboRhBloodUnitModel can't be null since it is checked earler (see above). 
            if (pendingAboRhBloodUnitModel.BloodUnitModel == null)
            {
                throw new Exception("Pending blood unit does not have a Product Code specified (Cannot GenerateExceptionUnitAboRh).");
            }
            //*** Fortify Justified Code ***

            var aboInterop = pendingAboRhBloodUnitModel.AboInterp == null ? "" : pendingAboRhBloodUnitModel.AboInterp.TestResultText;
            var rhInterp = pendingAboRhBloodUnitModel.RhInterp == null ? "" : pendingAboRhBloodUnitModel.RhInterp.TestResultText;

            exceptionReport.GenerateExceptionUnitAboRh(pendingAboRhBloodUnitModel.BloodUnitModel.BloodUnitGuid,
                                                       aboInterop,
                                                       rhInterp,
                                                       pendingAboRhBloodUnitModel.InstrumentUserId);

            return exceptionReport;
        }
    }
}
